#!/usr/bin/env python
# -*- coding: utf-8 -*-

# -----------------------------------------------------------------------------
#  PROJECT      : HOBBIT THE TRUE STORY IN PYTHON
#
#  MODULE       : hobbit_true_story.py
#  AUTHOR       : Dominique Rioual
#  DESCRIPTION  : main and single module
#
#  LOG          :
#  18/01/2026   V1.0    DR  First release
#
#  This game is a port to python from the MS-DOS adventure released by 
#  Milbus Software (2001). Original game was created by Fredrik Ramsberg
#  and Johan Berntsson.
#
#  Here is original information about the game :
#  ***  The first known attempt at MS-DOS adventuring    ***
#  *** Launched by Johbe and Frera on 30:th of March '93 ***
#  ***             Updated in December 2001              ***
#
#  ----------------------------------------------------------------------------
#  This is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  This software is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with 'Hobbit the true story in python'.
#  If not, see <https://www.gnu.org/licenses/>.
#
# -----------------------------------------------------------------------------


#------------------------------------------------------------------------------
# CONSTANTS
#------------------------------------------------------------------------------
# LOCATION
# --------
L_NONE     = -1

L_WOODS    = 0
L_FOUL     = 1
L_CAVE     = 2
L_CLEARING = 3
L_HOME     = 4
L_RIVEN    = 5
L_DEN      = 6
L_GANDALF  = 7
L_CHEST    = 8

L_ME       = 99

MAX_LOC_ID = 6


LOC_SHORT_DESC = ["Deep Forest", "Foul Smelling Cave", "Trolls' Cave", "Trolls' Clearing", "Tunnel Like Hall", "Rivendell", "Smaug's Den", \
                  "Gandalf", "Chest" ]
LOC_LONG_DESC = \
    [ "You are standing in the deep forest. Paths lead off in all directions but east,\n"\
      "where a mighty river is floating by. You feel lost and hungry.\n"\
      "Thorin sits down and starts singing about drugs. He offers you a joint.", \
      
      "You are in a vast cave, with a foul smell filling your nostrils. Light is\n"\
      "coming in through the wide entrance to the east, and it's just enough for you\n"\
      "to make out the walls of the cave.", \
      
      "You are in a hardly even lit cavern with a doorway leading south.\n"\
      "Thorin is mumbling to himself and fails to notice your presence.", \
      
      "You are standing in a vast clearing, with paths leading off to the east and\n"\
      "west. A yellow brick road leads north. Trolls are standing all around you,\n"\
      "fiercely watching every move of your limbs.", \
      
      "You are in your comfortable tunnel like hall. Infact, it's just a dirty hole in\n"\
      "the ground, but to you it's home. Gandalf is working on a nasty spell. There\n"\
      "is a round, green door set in the eastern wall.", \
      
      "This is Rivendell, the last homelike house. At least, that's the name of it.\n"\
      "Elrond is sitting at a round table together with his generals, discussing his\n"\
      "plans for a coming conquest and enslavement of the Hobbits. Paths lead off to\n"\
      "the north and west.\n"\
      "Elrond hesitatingly offers you some food. He gives your fairly thick legs a\n"\
      "greedy look. He drools.\n"\
      "Thorin announces: \"If you're attacking the Hobbits, count me in!\"", \
      
      "You are in a great cave. Gold and jewelry cover the floor. Precious paintings\n"\
      "are hanging on all the walls. Smaug is lying on a big heap of silver bars,\n"\
      "smoking a huge cigar. He looks at you and yawns loudly.", \
    
      "With Gandalf.",
       
      "In Chest." ]

# OBJECTS
#--------
O_NONE     = -1

O_TORCH    = 0
O_LUNCH    = 1
O_MAP      = 2
O_CHEST    = 3
O_RAFT     = 4
O_SWORD    = 5
O_TREASURE = 6
O_WDWARF   = 7
O_THORIN   = 8
O_ELROND   = 9
O_GANDALF  = 10
O_SPELL    = 11
O_SMAUG    = 12
O_PAINTING = 13
O_CIGAR    = 14
O_TROLL    = 15
O_NORTH    = 16
O_SOUTH    = 17
O_EAST     = 18
O_WEST     = 19
O_HOBOFF   = 20
O_PLANS    = 21

OBJ_NAME = [ "torch",   "lunch", "map",   "chest",    "raft",  "sword", "treasure", "woman", "thorin", "elrond", \
             "gandalf", "spell", "smaug", "painting", "cigar", "troll", "north", "south", "east", "west", "hoboff", "plans" ]

OBJ_DESC = [ "A Torch", "Some Lunch", "A Curious Map", "A Wooden Chest", "A Raft", "A Sword", "A Treasure", "A Naked Dwarf Woman", "Thorin", "Elrond",  \
             "Gandalf the gray", "A spell", "Smaug", "Beautiful paintings", "A cigar", "Ugly trolls", "North", "South", "East", "West", "HOBOFF", "Plans" ]

OBJ_CAN_TAKE = [ True, True, True, False, False, True, True, False, False, False, \
             False, False, False, False, False, False, False, False, False, False, False, False ]

NB_OBJ = len(OBJ_NAME)
MAX_OBJ_ID = NB_OBJ - 1

OBJ_SYNONYM = {"naked": O_WDWARF, "dragon": O_SMAUG, "paintings": O_PAINTING, "trolls": O_TROLL, "n": O_NORTH, "s": O_SOUTH , \
               "e" : O_EAST, "w":O_WEST, "plan":O_PLANS, "wizard":O_GANDALF, "magicien":O_GANDALF }

# OTHER CONSTANTS
# ---------------
FILE_NAME_BAK="hobbit.sav"

#------------------------------------------------------------------------------
# GLOBAL VARIABLES
#------------------------------------------------------------------------------
current_loc = L_HOME
hbs = False
game_over = False

obj_loc  = [L_CHEST, L_RIVEN, L_GANDALF, L_HOME, L_WOODS, L_CAVE, L_DEN, L_FOUL, L_ME, L_RIVEN, \
            L_HOME, L_GANDALF, L_DEN, L_DEN, L_DEN, L_CLEARING, L_NONE, L_NONE, L_NONE, L_NONE, L_GANDALF, L_NONE ]



#------------------------------------------------------------------------------
# FUNCTIONS
#------------------------------------------------------------------------------

#------------------------------------------------------------------------------
def show_clue():
    if current_loc == L_CLEARING:
        print("Trolls are not known to stand sunlight very well. It may be wise to wait a")
        print("while before confronting them.")
    else :
        "Try examining things!"
    
    return

#------------------------------------------------------------------------------
def do_cut(obj_id):
    global current_loc
    global obj_loc

    if obj_id == O_SMAUG or obj_id == O_THORIN or obj_id == O_ELROND or obj_id == O_TROLL :
        do_kill()
        return
    elif obj_id == O_CIGAR :
        if current_loc != L_DEN :
            print("There are no cigars around.")
            return
        if obj_loc[O_SWORD] != L_ME :
            print("You haven't got anything to cut with.")
            return

        print("You run towards the mighty dragon and cut off his cigar. The dragon")
        print("exclaims, \"Hey, what are you doing!\". You explain that the cigar will")
        print("be easier to smoke now, and after some experimenting, Smaug agrees with you.")
        print("He trades your sword for a treasure and offers you transport to Rivendell.")
        print("")
        current_loc = L_RIVEN
        obj_loc[O_TREASURE]=L_ME
        obj_loc[O_SWORD]=L_DEN
        pause()
        do_look()

    else :
        print("You can't cut that!")
         
    return

#------------------------------------------------------------------------------
def do_drop(obj_id):
    global obj_loc

    if (obj_id < 0 or obj_id > MAX_OBJ_ID) :
        print("Drop what ?")
        return

    if obj_loc[obj_id] == L_ME :
        obj_loc[obj_id] = current_loc
        print("You drop the", OBJ_NAME(obj_id))
    else :
        print("You are not holding that.")
    return

#------------------------------------------------------------------------------
def do_eat(obj_id):
    global obj_loc

    if (obj_id < 0 or obj_id > MAX_OBJ_ID):
        print("What do you want to eat?")
    elif (obj_id != O_LUNCH) :
        print("You can't eat that!")
    else :
        if obj_loc[O_LUNCH] == L_ME :
            print("You feel much refreshed.")
            obj_loc[O_LUNCH] = L_NONE
        else :
            print("But you don't have any",OBJ_NAME[O_LUNCH],"!")

    return


#------------------------------------------------------------------------------
def do_examine(obj_id):
    global obj_loc

    if obj_id < 0 or obj_id > MAX_OBJ_ID :
        print("Examine what ?")
        return 
    
    if obj_loc[obj_id] != L_ME and obj_loc[obj_id] != current_loc :
        if obj_id == O_GANDALF or obj_id == O_ELROND or obj_id == O_SMAUG :
            print(OBJ_DESC[obj_id], "isn't here!")
        elif obj_id == O_WDWARF : 
            print("She is nowhere to be seen.")
        elif obj_id == O_MAP and obj_loc[O_MAP] == L_GANDALF and current_loc == L_HOME :
            print("Gandalf has it. You should ask him about it. Maybe he doesn't need it anyway.")
        else :
            print("It is nowhere to be seen.")

        return
    
    if obj_id == O_TORCH :
        print("It is currently providing light.")

    elif obj_id == O_LUNCH :
        print("It's edible.")

    elif obj_id == O_MAP :
        print("The curious map must obviously have been drawn by hand, at great effort.")
        print("It shows the mysterious surroundings of Rivendell, home of the elves.")

    elif obj_id == O_CHEST :
        print("It's a nice piece of wooden workmanship.")
        if obj_loc[O_TORCH] == L_CHEST :
            print("There's a burning torch inside it. You decide to pick it up.")
            obj_loc[O_TORCH] = L_ME

    elif obj_id == O_GANDALF :
        print("Gandalf, the old magician, is still working with his new spell.")
        if obj_loc[O_MAP] == L_GANDALF :
            print("He is carrying a map.")

    elif obj_id == O_SPELL :
        print("It's a piece of beautiful yellowed paper, with magical symbols arranged")
        print("in groups. Several symbols resemble Hobbits exploding.")

    elif obj_id == O_ELROND :
        print("Elrond has turned pretty nasty lately. Better not mess with him, considering")
        print("his plans to enslave all Hobbits.")

    elif obj_id == O_THORIN :
        print("Thorin, your old friend, is no longer the proud dwarf he once was. All the")
        print("drugs have turned him into a drooling good-for-nothing idiot. The only reason")
        print("that he has come along is to get more treasures for drugs, as always.")
        if   current_loc == L_HOME :
            print("He doesn't seem too happy about leaving the house.")
        elif current_loc == L_RIVEN :
            print("Thorin takes another beer from the fridge. Elrond slaps him.")
        elif current_loc == L_DEN :
            print("Thorin is negotiating with the dragon about taking some rare drugs home,")
            print("provided that Smaug gets to keep your body.")
        elif current_loc == L_WOODS : 
            print("Thorin says \"We're lost! Abandon all hope. We\'ll never get out of here alive!\".")
        elif current_loc == L_CLEARING :
            if hbs == False :
                print("Thorin participates in the chanting.")
            else :
                print("Thorin is talking to the trolls. He doesn't notice that they are dead. ")
        elif current_loc == L_CAVE :
            print("Thorin is writing a song. I shall call it \"Let's roll another one\", he remarks.")

    elif obj_id == O_WDWARF :
        print("She looks cold and tired, and pretty far from sensual. As you peek at her,")
        print("Thorin remarks that she's the only dwarf woman alive at the moment. He seems")
        print("to find her attractive.")
    
    elif obj_id == O_SMAUG :
        print("Smaug's red dragon body is filling the eastern part of the cave. As he can't")
        print("get out of the cave anymore, he seems to have settled with just lying here,")
        print("eating any adventurers that drop by. Every now and then he enjoys a big")
        print("cigar, but he obviously doesn't know quite how to do it right. The cigar in")
        print("his mouth hasn't been cut properly. Smaug coughs, but luckily you're out of")
        print("the deadly range.")

    elif obj_id == O_TREASURE :
        print("It is just the kind of treasure you'd expect to find in a dragon's den. It does")
        print("seem mighty valuable.")
    
    elif obj_id == O_PAINTING :
        print("Among the paintings you recognize works of Picasso, da Vinci and Rembrandt.")

    elif obj_id == O_CIGAR :
        print("The cigar doesn't seem to be properly cut.")
    
    elif obj_id == O_TROLL :
        print(OBJ_DESC[obj_id])

    else :
        print("It is nowhere to be seen.")
    
    return

#------------------------------------------------------------------------------
def do_go(dir_id) :
    global current_loc
    global obj_loc
    global game_over
    previous_loc = current_loc

    if dir_id < 0 or dir_id > MAX_OBJ_ID :
        print("You can't go there.")
        return

    # move from woods
    if current_loc == L_WOODS :
        if dir_id == O_RAFT and obj_loc[O_RAFT] == L_WOODS :
            print("Not knowing what is about to happen, you fearlessly enter the raft. You follow")
            print("the river downstream for a while, as you don't have any means of manouvering")
            print("the little dingy. When hitting land, you are cold and starving. You soon find")
            print("a trail leading to a little village. The natives feed you and treat you well,")
            print("so you are soon healthy enough to go hunting on your own. On one of your trips")
            print("up into the mountains, you are suddenly brought back to your original mission,")
            print("as you step on a loose rock, which gives way. You slide down a long tunnel")
            print("which has many bends and little light. Eventually you leave the tunnel, only")
            print("to come into...\n")
            obj_loc[O_RAFT] = L_NONE
            current_loc = L_DEN
            pause()
            do_look()
            print("Thorin, whom you haven't seen since you were in the forest together, enters")
            print("from a side door, singing about peace. He greets you with a laugh.")
            return
        elif dir_id == O_SOUTH :
            current_loc = L_RIVEN
        elif dir_id == O_WEST :
            current_loc = L_FOUL
        elif dir_id == O_NORTH :
            current_loc = L_WOODS
        else :
             pass   #fail

            
    # move from foul
    elif current_loc == L_FOUL :
        if dir_id == O_EAST :
            current_loc = L_WOODS
        else :
            pass   #fail
    
    # move from cave
    elif current_loc == L_CAVE :
        if dir_id == O_SOUTH :
            current_loc = L_CLEARING
        else :
            pass   #fail

    # move from clearing
    elif current_loc == L_CLEARING :
        if dir_id == O_EAST :
            current_loc = L_RIVEN
        elif dir_id == O_NORTH :
            if hbs == True :
                current_loc = L_CAVE
            else :
                print("The trolls won't let you.")
                return
        elif dir_id == O_WEST :
            current_loc = L_HOME
            if obj_loc[O_TREASURE] == L_ME :
                do_look()
                print()
                print("As you enter your home again, Gandalf leaps to his feet. He comes forward, as")
                print("if to greet and congratulate you, but instead he asks, \"Did you bring any")
                print("tobacco?\". You decide to do what you should have done a long time ago. You")
                print("show the old wizard to the door, and kindly ask him to leave. Thorin follows")
                print("Gandalf out through the door.\n")
                print("Without any wizards or dwarves pestering you all the time, your life gets very")
                print("pleasant, and with the treasure it stays that way. You have won!!!\n")
                game_over = True
                return
        else : 
            pass    #fail

    # move from home
    elif current_loc == L_HOME :
        if dir_id == O_EAST :
            current_loc = L_CLEARING
        else :
            pass    #fail

    # move from Rivendell
    elif current_loc == L_RIVEN :
        if (dir_id == O_NORTH or dir_id == O_WEST) and obj_loc[O_LUNCH] == L_ME :
            print("Just as you leave, Elrond grabs your lunch. He looks annoyed.")
            obj_loc[O_LUNCH] == L_RIVEN
        
        if dir_id == O_NORTH :
            if obj_loc[O_MAP] == L_ME :
                print("Using the curious map that Gandalf gave you, you soon find your way to the")
                print("pleasant lush of the woods.")
                current_loc = L_WOODS
                do_look()
            else :
                print("You stagger off into the surrounding hills, but find nothing of interest.")
                print("Disappointed you return back to Elrond's party.")
            return
        elif dir_id == O_WEST :
            current_loc = L_CLEARING 
        else :
            pass    #fail

    # move from den
    elif current_loc == L_DEN :
        pass    #fail

    else :
        print("Error: Undefined room!")
        return
    
    if current_loc != previous_loc :
        print("You walk", OBJ_NAME[dir_id])
        print()
        do_look()
        if dir_id == O_NORTH :
            print("Thorin is already here.")
        elif dir_id == O_SOUTH :
            print("Thorin comes running after you.")
        elif dir_id == O_EAST :
            print("Thorin enters from the west.")
        elif dir_id == O_WEST :
            print("Thorin follows your trail.")
        else :
            pass
    else :
        print("You can't go", OBJ_NAME[dir_id])

    return


#------------------------------------------------------------------------------
def inventory():
    print("You are carrying:")
    for i in range(0, NB_OBJ) :
        if obj_loc[i] == L_ME and OBJ_CAN_TAKE[i] == True :
            print(OBJ_DESC[i])
    return

#------------------------------------------------------------------------------
def do_kill():
    print("Violence is not very likely to solve your problems. Better stay calm.")
    return


#------------------------------------------------------------------------------
def do_load():
    global current_loc
    global hbs
    global obj_loc

    try:
        f = open(FILE_NAME_BAK, "r")

    except :
        print("Impossible to load game backup due to system error.")
        return
    
    current_loc = int(f.readline())
    s = f.readline()
    if s == "True" :
        hbs = True
    else :
        hbs = False
    for i in range(0,MAX_LOC_ID) :
        obj_loc[i] = int(f.readline())

    print("Game backup loaded.")
    f.close()

    return


#------------------------------------------------------------------------------
def do_look():
    global obj_loc

    if current_loc < 0 or current_loc > MAX_LOC_ID :
        print("Error: Undefined room!")
        return

    if (current_loc == L_CAVE) and ((obj_loc[O_TORCH] != L_CAVE) and (obj_loc[O_TORCH] != L_ME)):
        print("It's dark. You can't see anything.")
        return
    
    print(LOC_SHORT_DESC[current_loc], ":")
    print(LOC_LONG_DESC[current_loc])    
     
    if current_loc == L_CLEARING :
        if hbs == True :
            print("The trolls seem to be stoned. They probably don't like the sunlight.")
        else :
            print("The trolls are chanting ancient rhymes. They tend to do that just before dawn.")

    if current_loc == L_RIVEN and obj_loc[O_LUNCH] == L_RIVEN :
        obj_loc[O_LUNCH] = L_ME

    print("You can see:")
    for i in range(0, NB_OBJ) :
        if obj_loc[i] == current_loc : print(OBJ_DESC[i])
    
    print("Thorin, the dwarf.")


#------------------------------------------------------------------------------
def do_save():
    try:
        f = open(FILE_NAME_BAK, "w")

    except :
        print("Impossible to save the game due to system error.")
        return
    
    f.write(str(current_loc))
    f.write("\n")
    f.write(str(hbs))
    f.write("\n")
    for i in obj_loc :
        f.write(str(i))
        f.write("\n")
    print("Game saved.")
    f.close()

    return


#------------------------------------------------------------------------------
def do_take(obj_id):
    global obj_loc
    global game_over

    if (obj_id < 0 or obj_id > MAX_OBJ_ID) :
        print("Take what ?")
        return
    
    if OBJ_CAN_TAKE[obj_id] == False :
        print("You can't take that.")
        return
    
    if obj_loc[obj_id] != current_loc :
        print("You can't see it here.")
        return
    
    if obj_loc[obj_id] == L_ME :
        print("You are already carrying that.")
        return

    if current_loc == L_DEN and obj_id == O_TREASURE :
        print("Smaug yield: \"Filthy little thief! You're trying to steal my treasure!!!\"")
        print("Smaug starts breathing fire and you turn into hashes... End of the adventure.\n")
        game_over = True
        return

    print("You take the", OBJ_NAME[obj_id])
    obj_loc[obj_id] = L_ME

    return


#------------------------------------------------------------------------------
def do_talk(person_id, about_id):
    global obj_loc
    global game_over

    if (person_id == O_GANDALF or person_id == O_ELROND) and obj_loc[person_id] != current_loc :
        print("You can't see him here")
        return
    if person_id == O_WDWARF and obj_loc[person_id] != current_loc :
        print("You can't see her here.")

    if person_id == O_GANDALF :
        if about_id == O_MAP :
            if obj_loc[O_MAP] == L_GANDALF :
                print("\"Oh, that map. There is absolutely nothing special about it at all. I guess")
                print("you could have it, if you want it.\", Gandalf declares and hands you the map.")
                obj_loc[O_MAP] = L_ME
            else :
                print("Gandalf yells, \"I already gave it to you, didn't I!!!\".")
                       
        elif about_id == O_SPELL :
            print("\"This will be a great spell. I shall name it 'HOBOFF', no doubt.\", Gandalf")
            print("chuckles.")

        elif about_id == O_HOBOFF :
            print("Gandalf gives you a mean smile, showing all his teeth. \"Any Hobbit of my")
            print("choice will be turned into a pile of dirt.\", he laughs, \"Only a few minor")
            print("adjustments left, and it will produce a great fertilizer.\". His eyes open")
            print("wide as he reaches the end of the sentence. He then looks at you for a good")
            print("fifteen seconds before he slowly turns away.")

        elif about_id == O_THORIN :
            print("\"It's a sordid story, really. He hasn't been himself in a long time. Or perhaps")
            print("he wasn't himself before, and now he is. Or maybe we haven't even seen his real")
            print("self. Let's talk about something else, shall we?\" says Gandalf and looks tired.")
        
        else :
            print("\"Yes, that's probably so\", he says.")
        return
       
    elif person_id == O_ELROND :
        if about_id == O_MAP :
            if obj_loc[O_MAP] == L_ME or obj_loc[O_MAP] == L_RIVEN :
                print("\"That's a fine map you've got there. We wouldn't want anything to HAPPEN to it,")
                print("now would we?\", Elrond hisses. He continues; \"Whatever you do, don't show it to")
                print("that crazy dwarf!\".")
            else :
                print("\"What map are you talking about, you little prat? You must be delirious!\",")
                print("Elrond snarls. He slaps you in the face.")
        elif about_id == O_PLANS :
            print("\"They are top secret. If you reveal anything you're dead meat, pipsqueak!\",")
            print("Elrond whispers. He look as though he meant it.")
        else :
            print("Elrond laughs at your silly question. He pats your head.")
        return

    elif person_id == O_THORIN :
        if about_id == O_MAP :
            if obj_loc[O_MAP] == L_FOUL or obj_loc[O_MAP] == L_ME :
                print("Thorin takes a quick look at the map. His eyes widen in rage as he reaches for")
                print("his battle axe. \"It was you all the time, wasn't it?!\", he cries out. With one")
                print("well placed blow he cleaves your skull. All is dark...")
                print("")
                game_over = True
            else :
                print("Thorin eyes you suspiciously. \"Don't try any tricks, boy. We both know that")
                print("there is no map here, now don't we?\", he asks. He doesn't seem too sure about")
                print("it himself.")

        elif about_id == O_WDWARF :
            if current_loc == L_FOUL :
                print("Thorin looks embarrased. \"I'd rather not talk about it. She's...She's...Never")
                print("mind, it's none of your business anyway,\" he says. He briefly touches the")
                print("handle of his axe, as if to make it clear that you had better leave it at that.")
            else :
                print("Thorin glances around. \"Whoa, where is she? Where's this woman you're talking")
                print("about?\" he says. You are forced to disappoint him.")
        else :
            if current_loc == L_CLEARING :
                print("Thorin gets a serious look on his face. \"Don't push your luck, kid!\", he says.")
            if current_loc == L_WOODS :
                print("\"Have you ever tried picking up your teeth with broken fingers?\",Thorin replies.")
            if current_loc == L_CAVE :
                print("\"I'm tired of killing for money. Let's go home!\", Thorin sighs.")
            if current_loc == L_RIVEN :
                print("Thorin is too absorbed by Elronds plans to pay attention.")
            if current_loc == L_DEN :
                print("\"Hell, this is a fortune!\", Thorin states as he walks around the cave.")
            if current_loc == L_HOME :
                print("\"Did your mom ever talk to you about death?\", Thorin asks you.")
        return
    
    elif person_id == O_WDWARF :
        if about_id == O_MAP :
            if obj_loc[O_MAP] == L_FOUL or obj_loc[O_MAP] == L_ME :
                print("She looks at your map and laughs heartily. You seem to have made her day.")
            else :
                print("She doesn't seem to understand what you are referring to.")
        elif about_id == O_THORIN :
            print("She says something that sounds like an insult. Since the two of you don't seem")
            print("to have any language in common, you can't tell for sure though. She pats")
            print("Thorin's head. Thorin slaps you.")
        else :
            print("She shudders and looks nervous.")
        return
    
    else :
        print("Action speaks louder than words!")
    
    return


#------------------------------------------------------------------------------
def do_wait():
    global hbs
    if current_loc == L_CLEARING and hbs == False :
        print("A new day dawns. The trolls seem rather surprised, and rigid with fear.")
        hbs = True
    else :
        print("Time passes...")
    
    return


#------------------------------------------------------------------------------
def do_action(action, obj_id1, obj_id2) :

    if   action == "ask" or action == "talk":
        do_talk(obj_id1, obj_id2)
    elif action == "clue" or action == "hint" or action == "help":
        show_clue()
    elif action == "cut" :
        do_cut(obj_id1)
    elif action == "drop" :
        do_drop(obj_id1)
    elif action == "eat" :
        do_eat(obj_id1)
    elif action == "examine" or action =="x" :
        do_examine(obj_id1)
    elif action == "go" or action =="enter" :
        do_go(obj_id1)
    elif action == "n" or action =="north" :
        do_go(O_NORTH)
    elif action == "s" or action =="south" :
        do_go(O_SOUTH)
    elif action == "e" or action =="east" :
        do_go(O_EAST)
    elif action == "w" or action =="west" :
        do_go(O_WEST)
    elif action == "inventory" or action == "inv" or action == "i" :
        inventory()
    elif action == "kill" :
        do_kill()
    elif action == "load" :
        do_load()
    elif action == "look" or action =="l":
        do_look()
    elif action == "save" :
        do_save()
    elif action == "take" or action =="get" :
        do_take(obj_id1)
    elif action == "wait" :
        do_wait()
    else :
        print("You cannot do that !")
    
    return


#------------------------------------------------------------------------------   
def get_object_id(obj_name) :
    obj_id = O_NONE
    i = 0

    for i in range(0, NB_OBJ) :
        if (obj_name == OBJ_NAME[i]) :
            obj_id = i

    # look for synonyms
    for k, val in OBJ_SYNONYM.items():
        if obj_name == k:
            obj_id = val

    return obj_id


#------------------------------------------------------------------------------
def parse_command(cmd) :

    cmd_list = cmd.lower().split()
    cmd_list_filtered = []

    for s in cmd_list :
        if s != "the" and s != "to"  and s != "about" and s !="at":
            cmd_list_filtered.append(s)
    
    if len(cmd_list_filtered) == 1 :
        return(cmd_list_filtered[0], O_NONE, O_NONE)
    
    if len(cmd_list_filtered) == 2 :
        return( cmd_list_filtered[0], get_object_id(cmd_list_filtered[1]), O_NONE )
    
    if len(cmd_list_filtered) == 3 :
        return( cmd_list_filtered[0], get_object_id(cmd_list_filtered[1]), get_object_id(cmd_list_filtered[2]) )
    
    return("", O_NONE, O_NONE)
    

#------------------------------------------------------------------------------
def pause():
    x = input("Press 'Return' to continue")
    print("")
    return

#------------------------------------------------------------------------------
def intro():
    print("          _______  ______   ______  __________________")
    print("|\     /|(  ___  )(  ___ \ (  ___ \ \__   __/\__   __/")
    print("| )   ( || (   ) || (   ) )| (   ) )   ) (      ) (   ")
    print("| (___) || |   | || (__/ / | (__/ /    | |      | |   ")
    print("|  ___  || |   | ||  __ (  |  __ (     | |      | |   ")
    print("| (   ) || |   | || (  \ \ | (  \ \    | |      | |   ")
    print("| )   ( || (___) || )___) )| )___) )___) (___   | |   ")
    print("|/     \|(_______)|/ \___/ |/ \___/ \_______/   )_(   ")
    print()                                                                                          
    print("       The True Story - redux, Director's Cut.")
    print()
    print("An MS-DOS adventure released by Milbus Software (tm) 1993.")
    print("redux, Director's Cut released by Milbus Software (tm) 2001.")
    print("Python version released by Dominique Rioual - 2026.")

    return


#------------------------------------------------------------------------------
#------------------------------------------------------------------------------
# Main 
#------------------------------------------------------------------------------
#------------------------------------------------------------------------------
intro()
print()
do_look()
while(game_over == False):
    print()
    cmd = input("What do you do? ")
    (verb, obj_id1, obj_id2) = parse_command(cmd)

    if verb != "" :
        print()
        do_action(verb, obj_id1, obj_id2)

pause()
print("\n\n\n")
#------------------------------------------------------------------------------
# end of main 


def main(args):
    return 0

if __name__ == '__main__':
    import sys
    sys.exit(main(sys.argv))